<?php
session_start();
require 'vendor/autoload.php';  
require('pdf_header.php');

include_once 'model/stock_model.php';
include_once 'model/common_model.php';
include_once 'model/branch_model.php';

$common_model = new common_model();
$stock_model = new stock_model();
$branch_model = new branch_model();

// Get logo dynamically
$logo = $common_model->getlogo();
$logo_path = '../finance/images/'.$logo;
$company_name = $common_model->getCompanyName();
$company_details = $common_model->getCompanyDetails();
$company_address = $company_details['address'];
$country = $company_details['country'];
$phone_number = $company_details['phone_number'];
$email = $company_details['email'];
$company_website = $company_details['website'];

if(isset($_POST['stock_sheets_pdf'])) {
    $start_date = $_POST['start_date'];
    $end_date = $_POST['end_date'];
    $location = $_POST['location'];

    $stock_sheet = $stock_model->getstockSheet($location, $start_date, $end_date);
    $branchDetails = $branch_model->getBranchDetails($location);

    if(!empty($stock_sheet)) {

        // 1️⃣ Filter items with opening stock > 0
        $stock_sheet = array_filter($stock_sheet, function ($item) {
            return (int)$item['opening_stock'] > 0;
        });

        // 2️⃣ Sort by custom category order, then by product name
        $categoryOrder = [
            'CAR BATTERIES' => 1,
            'SOLAR BATTERIES' => 2,
            'INVERTERS' => 3,
            'SOLAR PANELS' => 4,
            'ACCESSORIES' => 5
        ];

        usort($stock_sheet, function ($a, $b) use ($categoryOrder) {
            $catA = $categoryOrder[$a['categoryname']] ?? 999;
            $catB = $categoryOrder[$b['categoryname']] ?? 999;

            if ($catA === $catB) {
                return strcmp($a['product_name'], $b['product_name']); // alphabetically within category
            }
            return $catA <=> $catB;
        });

        // Initialize PDF
        $pdf = new FPDF('L','mm','A4'); // Landscape A4
        $pdf->AddPage();

        // Company / Warehouse header
        addPDFHeader(
            $pdf,
            $branchDetails['branch_name'],
            $logo_path,
            $company_address,
            $country,
            $phone_number,
            $email,
            $company_website
        );

        $pdf->Ln(5);

        // Stock sheet title + date
        $pdf->SetFont('Arial', 'B', 12);
        $pdf->Cell(0, 8, "Branch Stock Sheet: " . $branchDetails['branch_name'], 0, 1, 'L');

        $pdf->SetFont('Arial', '', 10);
        $pdf->Cell(0, 6, "Date: " . date('d-m-Y'), 0, 1, 'R');

        // Selected period
        $pdf->SetFont('Arial', 'I', 10);
        $pdf->Cell(0, 6, "Period: " . date('d-m-Y', strtotime($start_date)) . " to " . date('d-m-Y', strtotime($end_date)), 0, 1, 'L');

        $pdf->Ln(3);

        // Table header
        $pdf->SetFont('Courier', 'B', 9);
        $pdf->SetFillColor(200,200,200);

        $pdf->Cell(40, 7, 'Product', 1, 0, 'L', true);
        $pdf->Cell(30, 7, 'Category', 1, 0, 'L', true);
        $pdf->Cell(25, 7, 'Opening', 1, 0, 'C', true);
        $pdf->Cell(25, 7, 'Transfer In', 1, 0, 'C', true);
        $pdf->Cell(25, 7, 'Total In', 1, 0, 'C', true);
        $pdf->Cell(25, 7, 'Transfer Out', 1, 0, 'C', true);
        $pdf->Cell(25, 7, 'Sold', 1, 0, 'C', true);
        $pdf->Cell(25, 7, 'Total Out', 1, 0, 'C', true);
        $pdf->Cell(25, 7, 'Closing', 1, 0, 'C', true);
        $pdf->Cell(25, 7, 'Current', 1, 1, 'C', true);

        // Table rows
        $pdf->SetFont('Courier', '', 8);
        $pdf->SetLineWidth(0.1);

        foreach($stock_sheet as $item) {
            $opening = (int)$item['opening_stock'];
            $transfer_in = (int)$item['transfer_in'];
            $transfer_out = (int)$item['transfer_out'];
            $sold = (int)$item['sold'];

            $total_in = $opening + $transfer_in;
            $total_out = $transfer_out + $sold;
            $closing_stock = $total_in - $total_out;
            $current_stock = (int)$item['current_qty'];

            // Conditional row coloring
            if($current_stock == 0) {
                $pdf->SetFillColor(255, 0, 0); // red
                $fill = true;
            } elseif($current_stock < $item['min_level']) {
                $pdf->SetFillColor(255, 255, 0); // yellow
                $fill = true;
            } else {
                $fill = false;
            }

            $pdf->Cell(40, 6, $item['product_name'], 1, 0, 'L', $fill);
            $pdf->Cell(30, 6, $item['categoryname'], 1, 0, 'L', $fill);
            $pdf->Cell(25, 6, $opening, 1, 0, 'C', $fill);
            $pdf->Cell(25, 6, $transfer_in, 1, 0, 'C', $fill);
            $pdf->Cell(25, 6, $total_in, 1, 0, 'C', $fill);
            $pdf->Cell(25, 6, $transfer_out, 1, 0, 'C', $fill);
            $pdf->Cell(25, 6, $sold, 1, 0, 'C', $fill);
            $pdf->Cell(25, 6, $total_out, 1, 0, 'C', $fill);
            $pdf->Cell(25, 6, $closing_stock, 1, 0, 'C', $fill);
            $pdf->Cell(25, 6, $current_stock, 1, 1, 'C', $fill);
        }

        // Filename
        $branchName = str_replace(' ', '_', $branchDetails['branch_name']);
        $dateTime = date('Y-m-d_H-i-s');
        $filename = "Branch_Stock_sheet_{$branchName}_{$dateTime}.pdf";

        $pdf->Output('D', $filename);
    }
}
