<?php
    session_start();
    include_once '../model/employees_model.php';

    $employees_model = new employees_model();

    // Check if form was submitted and action is set
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
        $post_action = $_POST['action'];

        switch ($post_action) {
            case 'update_salary':
                changeEmployeeSalary($employees_model);
                break;

            case 'change_state':
                changeEmployeeState($employees_model);
                break;

            case 'update_employee_details':
                updateEmployeeDetails($employees_model);
                break;

            default:
                $_SESSION['notification'] = [
                    'type' => 'error',
                    'title' => 'Unknown Action',
                    'message' => 'The requested action is not recognized.'
                ];
                header('Location: ../manage_employees.php');
                exit();
        }


    }

    function updateEmployeeDetails($employees_model) {
        // Sanitize inputs
        $employee_code      = htmlspecialchars($_POST['employee_code'] ?? '', ENT_QUOTES, 'UTF-8');
        $national_id_number = htmlspecialchars($_POST['national_id_number'] ?? '', ENT_QUOTES, 'UTF-8');
        $fullname           = htmlspecialchars($_POST['fullname'] ?? '', ENT_QUOTES, 'UTF-8');
        $phone_number       = htmlspecialchars($_POST['phone_number'] ?? '', ENT_QUOTES, 'UTF-8');
        $email_address      = filter_var(trim($_POST['email_address'] ?? ''), FILTER_SANITIZE_EMAIL);
        $gender             = htmlspecialchars($_POST['gender'] ?? '', ENT_QUOTES, 'UTF-8');
        $date_of_birth      = htmlspecialchars($_POST['date_of_birth'] ?? '', ENT_QUOTES, 'UTF-8');
        $current_location   = htmlspecialchars($_POST['current_location'] ?? '', ENT_QUOTES, 'UTF-8');
        $nationality        = htmlspecialchars($_POST['nationality'] ?? '', ENT_QUOTES, 'UTF-8');
        $home_village       = htmlspecialchars($_POST['home_village'] ?? '', ENT_QUOTES, 'UTF-8');
        $home_district      = htmlspecialchars($_POST['home_district'] ?? '', ENT_QUOTES, 'UTF-8');

        // Check required fields
        $required_fields = [
            'employee_code', 'national_id_number', 'fullname', 'phone_number',
            'email_address', 'gender', 'date_of_birth',
            'current_location', 'nationality', 'home_village', 'home_district'
        ];

        foreach ($required_fields as $field) {
            if (empty($_POST[$field])) {
                $_SESSION['notification'] = [
                    'type' => 'error',
                    'title' => 'Validation Error',
                    'message' => 'All fields are required.'
                ];
                header('Location: ../manage_employees.php');
                exit();
            }
        }

        // Check for duplicate email
        if ($employees_model->emailExistsForOtherEmployee($email_address, $employee_code)) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Duplicate Email',
                'message' => 'This email address is already in use by another employee.'
            ];
            header('Location: ../manage_employees.php');
            exit();
        }

        // Check for duplicate National ID
        if ($employees_model->nationalIDExistsForOtherEmployee($national_id_number, $employee_code)) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Duplicate National ID',
                'message' => 'This National ID Number is already registered with another employee.'
            ];
            header('Location: ../manage_employees.php');
            exit();
        }

        // Prepare sanitized data
        $data = [
            'employee_code'        => $employee_code,
            'national_id_number'   => $national_id_number,
            'fullname'             => $fullname,
            'phone_number'         => $phone_number,
            'email_address'        => $email_address,
            'gender'               => $gender,
            'date_of_birth'        => $date_of_birth,
            'current_location'     => $current_location,
            'nationality'          => $nationality,
            'home_village'         => $home_village,
            'home_district'        => $home_district
        ];

        // Update in DB
        $updated = $employees_model->updateEmployeeDetails($data);

        // Notify and redirect
        $_SESSION['notification'] = [
            'type' => $updated ? 'success' : 'error',
            'title' => $updated ? 'SUCCESS' : 'FAILED!!',
            'message' => $updated
                ? 'Employee details were updated successfully.'
                : 'Failed to update employee details. Please try again.'
        ];

        header('Location: ../employee_details.php?employee_code='.$employee_code);
        exit();
    }


    function changeEmployeeSalary($employees_model) {
        $employee_code = (int) ($_POST['employee_code'] ?? 0);
        $new_salary = $_POST['new_salary'];

        // Check for missing values
        if (!$employee_code || !$new_salary) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'REQUEST FAILED',
                'message' => 'Missing required values for processing salaries.'
            ];
            header('Location: ../employee_details.php?employee_code='.$employee_code);
            exit;
        }

        // Check if new salary is numeric
        if (!is_numeric($new_salary) || $new_salary <= 70000) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Invalid Salary',
                'message' => 'Please provide a valid salary amount.'
            ];
            header('Location: ../employee_details.php?employee_code='.$employee_code);
            exit;
        }

        // Optional: Check salary range (e.g., between 500 and 50,000)
        $min_salary = 70000;
        $max_salary = 1500000;
            
        if ($new_salary < $min_salary || $new_salary > $max_salary) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Salary Out of Range',
                'message' => "The salary must be between MK {$min_salary} and MK {$max_salary}."
            ];
            header('Location: ../employee_details.php?employee_code='.$employee_code);
            exit;
        }

        // Check if employee exists
        $exempted_codes = [2000, 2001, 2002, 2003];
        if (!in_array($employee_code, $exempted_codes)) {
            $user = $employees_model->getEmployeeDetails($employee_code);

            if (!$user) {
                $_SESSION['notification'] = [
                    'type' => 'error',
                    'title' => 'User Not Found',
                    'message' => 'User record not found.'
                ];
                header('Location: ../employee_details.php?employee_code='.$employee_code);
                exit();
            }

            // Check if the employee is active (not suspended or terminated)
            if ($user['employee_state'] == 0) {
                $_SESSION['notification'] = [
                    'type' => 'error',
                    'title' => 'Employee Suspended',
                    'message' => 'The employee is suspended and cannot have their salary changed.'
                ];
                header('Location: ../employee_details.php?employee_code='.$employee_code);
                exit;
            }
        }

        // If all checks pass, call the model method to update the salary
        $result = $employees_model->updateEmployeeSalary($employee_code, $new_salary);
            
        if ($result) {
            $_SESSION['notification'] = [
                'type' => 'success',
                'title' => 'Salary Updated',
                'message' => 'The salary has been successfully updated.'
            ];
        } else {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Update Failed',
                'message' => 'Failed to update the salary. Please try again.'
            ];
        }
            
        // Redirect to the employee management page
        header('Location: ../employee_details.php?employee_code='.$employee_code);
        exit();
    }

    function changeEmployeeState($employees_model) {
        
        $employee_code = (int) ($_POST['employee_code'] ?? 0);
        $employee_state = $_POST['value']; 

        
        if (!$employee_code || !isset($employee_state)) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Missing Required Information',
                'message' => 'Employee code or action (resume/suspend) is missing.'
            ];
            header('Location: ../employee_details.php?employee_code='.$employee_code);
            exit;
        }

        $employee = $employees_model->getEmployeeDetails($employee_code);

        if (!$employee) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Employee Not Found',
                'message' => 'Employee with the provided code does not exist.'
            ];
            header("Location: ../manage_employees.php");
            exit;
        }

        // Ensure that we're not trying to suspend an already suspended employee
        if ($employee['employee_state'] == 0 && $employee_state == '0') {
            // Attempting to suspend an already active employee
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Employee Already Active',
                'message' => 'This employee is already suspended.'
            ];
           header('Location: ../employee_details.php?employee_code='.$employee_code);
            exit;
        }

        if ($employee['employee_state'] == 1 && $employee_state == '1') {
            // Attempting to resume an already resumed  employee
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Employee Already Active',
                'message' => 'This employee is already in active state.'
            ];
            header('Location: ../employee_details.php?employee_code='.$employee_code);
            exit;
        }

        // Proceed to update employee state (suspend or resume)
        $result = $employees_model->updateEmployeeState($employee_code, $employee_state);

        // Check if the update was successful
        if ($result) {
            // Success - Employee state updated
            $_SESSION['notification'] = [
                'type' => 'success',
                'title' => $employee_state == '1' ? 'Employee Resumed' : 'Employee Suspended',
                'message' => 'The employee state has been successfully updated.'
            ];
        } else {
            // Failure - Could not update employee state
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Update Failed',
                'message' => 'Failed to update the employee state. Please try again later.'
            ];
        }

        // Redirect back to manage employees page
        header('Location: ../employee_details.php?employee_code='.$employee_code);
        exit();
    }



