<?php

include_once 'config.php';

class employees_model
{
    public $link;

    public function __construct()
    {
        $db_connection = new dbConnection();
        $this->link = $db_connection->connect();
    }

    public function getWarehouseNameByEmpCode($employee_code) {
        $stmt = $this->link->prepare("SELECT warehouse_name FROM warehouses WHERE supervisor = :employee_code LIMIT 1");
        $stmt->bindParam(':employee_code', $employee_code, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchColumn(); // returns the name or false
    }

    public function getBranchNameByEmpCode($employee_code) {
        $stmt = $this->link->prepare("SELECT branch_name FROM branches WHERE shop_manager = :employee_code LIMIT 1");
        $stmt->bindParam(':employee_code', $employee_code, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchColumn();
    }



    public function updateEmployeeState($employee_code, $employee_state) {
        if ($employee_state !== '0' && $employee_state !== '1') {
            return false; // Invalid state
        }

        // Update the employee state in the database
        $sql = "UPDATE employees SET employee_state = :employee_state WHERE employee_code = :employee_code";

        try {
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':employee_state', $employee_state, PDO::PARAM_INT);
            $stmt->bindParam(':employee_code', $employee_code, PDO::PARAM_INT);
            return $stmt->execute(); // Return true if the update was successful
        } catch (PDOException $e) {
            return false; // If an error occurs, return false
        }
    }

    public function getUserByEmployeeCode(int $employee_code): ?array {
        $stmt = $this->link->prepare("SELECT * FROM employees WHERE employee_code = :employee_code LIMIT 1");
        $stmt->bindParam(':employee_code', $employee_code, PDO::PARAM_INT);
        $stmt->execute();

        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        return $user ?: null;
    }

    public function getRoleDetails($employee_code) {
        $roles = [];
        $location = null;
        $location_id = null;

        // Check Supervisor role
        $sql1 = "SELECT 
                ll.location_name,
                ll.loc_id
            FROM 
                warehouses w
            LEFT JOIN
                locations ll ON w.location = ll.loc_id 
            WHERE 
                w.supervisor = :employee_code LIMIT 1";
        $stmt1 = $this->link->prepare($sql1);
        $stmt1->execute(['employee_code' => $employee_code]);
        $warehouse = $stmt1->fetch(PDO::FETCH_ASSOC);
        if ($warehouse) {
            $roles[] = 'Supervisor';
            $location = $warehouse['location_name'];
            $location_id = $warehouse['loc_id'];
        }

        // Check Cashier role
        $sql2 = "SELECT 
                    ll.location_name,
                    ll.loc_id
                FROM 
                    branches b
                LEFT JOIN
                    locations ll ON b.location = ll.loc_id
                WHERE 
                    b.shop_manager = :employee_code LIMIT 1";
        $stmt2 = $this->link->prepare($sql2);
        $stmt2->execute(['employee_code' => $employee_code]);
        $branch = $stmt2->fetch(PDO::FETCH_ASSOC);
        if ($branch) {
            $roles[] = 'Cashier';
            if (!$location) {
                $location = $branch['location_name'];
                $location_id = $branch['loc_id'];
            }
        }

        return [
            'roles' => $roles,
            'location' => $location,
            'location_id' => $location_id
        ];
    }




    public function getActiveEmployees() {
        $sql = "SELECT 
                    emp.fullname,
                    emp.national_id_number,
                    emp.phone_number,
                    emp.current_location,
                    emp.employee_code,
                    pp.postname,
                    dd.department_name,
                    emp.email_address,
                    ll.location_name,
                    emp.gender
                FROM 
                    employees emp
                LEFT JOIN
                    posts pp ON emp.post = pp.id
                LEFT JOIN
                    departments dd ON pp.department = dd.id
                LEFT JOIN
                    locations ll ON emp.current_location = ll.loc_id
                WHERE 
                    emp.employee_state = 1";
    
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
    
        return $stmt->fetchAll(PDO::FETCH_ASSOC); // Returns all active employees
    }

    public function isActiveEmployee($employee_code) {
        $sql = "SELECT COUNT(*) FROM employees WHERE employee_code = :employee_code AND employee_state = 1";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':employee_code', $employee_code, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchColumn() > 0;
    }

        public function getTodayBirthdayEmployees($date_today) {
        $sql = "SELECT 
                    emp.fullname, 
                    emp.employee_code, 
                    emp.date_of_birth, 
                    emp.start_date,
                    pp.postname as position
                FROM 
                    employees emp
                LEFT JOIN
                    posts pp ON emp.post = pp.id  
                WHERE DATE_FORMAT(date_of_birth, '%m-%d') = :today 
                AND employee_state = 1";
        
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':today', $date_today, PDO::PARAM_STR);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }


    public function getTodaysBirthdays() {
        $today = date('m-d');
        $sql = "SELECT fullname, employee_code 
            FROM 
                employees 
            WHERE 
                DATE_FORMAT(date_of_birth, '%m-%d') = :today 
                AND employee_state = 1";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':today', $today, PDO::PARAM_STR);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }



    public function getSupervisors() {
        $sql = "SELECT 
                emp.fullname,
                emp.phone_number,
                emp.current_location,
                emp.employee_code
            FROM 
                employees emp
            WHERE 
                emp.employee_state = 1
                AND emp.employee_code NOT IN (
                    SELECT employee_code FROM users WHERE role IN ('Supervisor', 'Cashier')
                )";
    
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
    
        return $stmt->fetchAll(PDO::FETCH_ASSOC); // Returns all active employees
    }

    public function getShopManagers() {
        $sql = "SELECT 
                emp.fullname,
                emp.phone_number,
                emp.current_location,
                emp.employee_code
            FROM 
                employees emp
            WHERE 
                emp.employee_state = 1
                AND emp.employee_code NOT IN (
                    SELECT shop_manager FROM branches WHERE shop_manager IS NOT NULL
                )";
    
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
    
        return $stmt->fetchAll(PDO::FETCH_ASSOC); // Returns all active employees
    }

    public function isWarehouseSupervisor(int $employee_code): bool {
        $stmt = $this->link->prepare("SELECT 1 FROM users WHERE role='Supervisor' AND employee_code = :employee_code LIMIT 1");
        $stmt->bindParam(':employee_code', $employee_code, PDO::PARAM_INT);
        $stmt->execute();
        return (bool) $stmt->fetchColumn();
    }

    public function isBranchManager(int $employee_code): bool {
        $stmt = $this->link->prepare("SELECT 1 FROM users WHERE role = 'Cashier' AND employee_code = :employee_code LIMIT 1");
        $stmt->bindParam(':employee_code', $employee_code, PDO::PARAM_INT);
        $stmt->execute();
        return (bool) $stmt->fetchColumn();
    }

    public function newleaveformula($employee_code) {
        $sql = "SELECT 
                    e.employee_code,
                    e.fullname,
                    e.start_date
                FROM employees e
                WHERE e.employee_code = :employee_code
                LIMIT 1";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':employee_code', $employee_code, PDO::PARAM_INT);
        $stmt->execute();
        $employee = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$employee) return null;

        $startDate = new DateTime($employee['start_date']);
        
        // Add 3-month probation period
        $probationEnd = (clone $startDate)->modify('+3 months');

        $today = new DateTime();
        if ($today < $probationEnd) {
            $monthsWorked = 0;
        } else {
            // Calculate full months after probation
            $interval = $probationEnd->diff($today);
            $monthsWorked = ($interval->y * 12) + $interval->m;
        }

        $totalDays = round($monthsWorked * 1.5, 1);

        // Get total leave taken
        $sql2 = "SELECT 
                    COALESCE(SUM(DATEDIFF(lr.approved_end_date, lr.approved_start_date) + 1), 0) AS total_taken
                FROM leave_requests lr 
                WHERE lr.employee = :employee_code
                AND lr.leaave_state = 1
                AND lr.leave_type = 'Annual Leave'";

        $stmt2 = $this->link->prepare($sql2);
        $stmt2->bindParam(':employee_code', $employee_code, PDO::PARAM_INT);
        $stmt2->execute();
        $leaveTaken = $stmt2->fetchColumn();

        $totalLeft = max($totalDays - $leaveTaken, 0);

        return [
            'employee_code' => $employee['employee_code'],
            'fullname' => $employee['fullname'],
            'start_date' => $employee['start_date'],
            'total_accrued' => $totalDays,
            'total_taken' => $leaveTaken,
            'total_left' => $totalLeft
        ];
    }


    public function getIndividualLeaveData($employee_code) {
        $sql = "SELECT 
                e.employee_code,
                e.fullname,
                (YEAR(CURDATE()) - YEAR(e.start_date) + 1) AS years_worked,
                (YEAR(CURDATE()) - YEAR(e.start_date) + 1) * 18 AS total_days,
                COALESCE(SUM(DATEDIFF(lr.approved_end_date, lr.approved_start_date) + 1), 0) AS total_taken,
                ((YEAR(CURDATE()) - YEAR(e.start_date) + 1) * 18 -
                COALESCE(SUM(DATEDIFF(lr.approved_end_date, lr.approved_start_date) + 1), 0)) AS total_left
            FROM 
                employees e
            LEFT JOIN 
                leave_requests lr 
                ON lr.employee = e.employee_code
                AND lr.leaave_state = 1
                AND lr.leave_type = 'Annual Leave'
            WHERE
                e.employee_code = :employee_code
            GROUP BY 
                e.employee_code, e.fullname, e.start_date
            LIMIT 1";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':employee_code', $employee_code, PDO::PARAM_STR);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }




    public function getEmployeesLeaveData($start, $end) {
        $sql = "SELECT 
                e.employee_code,
                e.fullname,
                (YEAR(CURDATE()) - YEAR(e.start_date) + 1) AS years_worked,
                (YEAR(CURDATE()) - YEAR(e.start_date) + 1) * 18 AS total_days,
                COALESCE(SUM(DATEDIFF(lr.approved_end_date, lr.approved_start_date) + 1), 0) AS total_taken,
                ((YEAR(CURDATE()) - YEAR(e.start_date) + 1) * 18 -
                COALESCE(SUM(DATEDIFF(lr.approved_end_date, lr.approved_start_date) + 1), 0)) AS total_left
            FROM 
                employees e
            LEFT JOIN 
                leave_requests lr 
                ON lr.employee = e.employee_code
                AND lr.leaave_state = 1
                AND lr.leave_type = 'Annual Leave'
            WHERE
                e.employee_state = 1
            GROUP BY 
                e.employee_code, e.fullname, e.start_date
            ORDER BY 
                total_days DESC";

        $stmt = $this->link->prepare($sql);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function newLeavecalculation($start, $end) {
        $sql = "SELECT 
                    e.employee_code,
                    e.fullname,
                    e.start_date,
                    COALESCE(SUM(DATEDIFF(lr.approved_end_date, lr.approved_start_date) + 1), 0) AS total_taken
                FROM 
                    employees e
                LEFT JOIN 
                    leave_requests lr 
                    ON lr.employee = e.employee_code
                    AND lr.leaave_state = 1
                    AND lr.leave_type = 'Annual Leave'
                WHERE
                    e.employee_state = 1
                GROUP BY 
                    e.employee_code, e.fullname, e.start_date
                ORDER BY 
                    e.start_date ASC";

        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $result = [];
        foreach ($rows as $row) {
            $startDate = new DateTime($row['start_date']);
            $now = new DateTime();

            // Calculate probation end date
            $probationEnd = (clone $startDate)->modify('+3 months');

            if ($probationEnd > $now) {
                $accruedDays = 0; // Still on probation
            } else {
                // Months after probation
                $monthsWorked = ($now->diff($probationEnd)->y * 12) + $now->diff($probationEnd)->m;
                $accruedDays = round($monthsWorked * 1.5, 1); // 1.5 leave days per month
            }

            $totalTaken = (float) $row['total_taken'];
            $totalLeft = max(0, $accruedDays - $totalTaken);

            $result[] = [
                'employee_code' => $row['employee_code'],
                'fullname' => $row['fullname'],
                'start_date' => $row['start_date'],
                'total_taken' => $totalTaken,
                'total_days' => $accruedDays,
                'total_left' => $totalLeft
            ];
        }

        return $result;
    }


    public function getleaveSummaryData($employee_code) {
    // Step 1: Fetch employee info + approved leave days
        $sql = "SELECT 
                    e.employee_code,
                    e.fullname,
                    e.start_date,
                    COALESCE(SUM(DATEDIFF(lr.approved_end_date, lr.approved_start_date) + 1), 0) AS total_approved
                FROM 
                    employees e
                LEFT JOIN 
                    leave_requests lr 
                    ON lr.employee = e.employee_code 
                    AND lr.leaave_state = 1
                    AND lr.leave_type = 'Annual Leave'
                WHERE 
                    e.employee_code = :employee_code
                GROUP BY 
                    e.employee_code, e.fullname, e.start_date";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':employee_code', $employee_code, PDO::PARAM_STR);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$row) return null;

        // Step 2: Calculate months worked after probation
        $startDate = new DateTime($row['start_date']);
        $now = new DateTime();

        $probationEnd = (clone $startDate)->modify('+3 months');

        if ($probationEnd > $now) {
            $monthsWorked = 0;
        } else {
            $interval = $now->diff($probationEnd);
            $monthsWorked = ($interval->y * 12) + $interval->m;
        }

        $accrued = round($monthsWorked * 1.5, 1);
        $taken = (float) $row['total_approved'];
        $left = max(0, $accrued - $taken);

        // Calculate total full calendar years for reference
        $yearsWorked = $startDate->diff($now)->y + 1;

        return [
            'employee_code'    => $row['employee_code'],
            'fullname'         => $row['fullname'],
            'start_date'       => $row['start_date'],
            'years_worked'     => $yearsWorked,
            'total_allocated'  => $accrued,
            'total_approved'   => $taken,
            'total_left'       => $left
        ];
    }


    public function saveLeaveRequest($leave_id, $leave_type, $employee, $start_date, $end_date, $requested_numdays, $datecaptured, $operator, $leave_state) {
        $sql = "INSERT INTO leave_requests (leave_id, leave_type, employee, start_date, end_date, requested_numdays, datecaptured, operator, leaave_state) VALUES (:leave_id, :leave_type, :employee, :start_date, :end_date, :requested_numdays, :datecaptured, :operator, :leave_state)";

        $stmt = $this->link->prepare($sql);

        $stmt->bindParam(':leave_id', $leave_id);
        $stmt->bindParam(':leave_type', $leave_type);
        $stmt->bindParam(':employee', $employee);
        $stmt->bindParam(':start_date', $start_date);
        $stmt->bindParam(':end_date', $end_date);
        $stmt->bindParam(':requested_numdays', $requested_numdays);
        $stmt->bindParam(':datecaptured', $datecaptured);
        $stmt->bindParam(':operator', $operator);
        $stmt->bindParam(':leave_state', $leave_state);

        try {
            $stmt->execute();
            return $this->link->lastInsertId();            
        } catch (PDOException $e) {
            
        }
    }

    public function approveLeaveRequest($requestId, $startDate, $endDate, $numDays, $leave_state, $approvedBy, $approvedDate) {
        $stmt = $this->link->prepare("UPDATE leave_requests 
            SET 
                approved_start_date = ?, 
                approved_end_date = ?, 
                approved_numdays = ?, 
                approved_by = ?, 
                approved_date = ?,
                leaave_state = ?
            WHERE id = ?");
        return $stmt->execute([$startDate, $endDate, $numDays, $approvedBy, $approvedDate, $leave_state, $requestId]);
    }

    public function getleaveRequests(){
        $sql = "SELECT 
                lr.id,
                emp.fullname,
                emp.employee_code,
                COALESCE(em.fullname,'Default Account') as operator,
                lr.leave_id,
                la.leave_name,
                la.leave_type,
                lr.start_date AS requested_start_date,
                lr.end_date AS requested_end_date,
                lr.requested_numdays,

                -- 🎯 Total allocated days (18 per year since joining)
                (TIMESTAMPDIFF(YEAR, emp.start_date, CURDATE()) + 1) * 18 AS total_days_allocated,

                -- 🎯 Total approved leave days
                (
                    SELECT 
                        COALESCE(SUM(DATEDIFF(approved_end_date, approved_start_date) + 1), 0)
                    FROM 
                        leave_requests lr2
                    WHERE 
                        lr2.employee = emp.employee_code
                        AND lr2.leaave_state = 1
                ) AS approved_days_taken,

                -- 🎯 Remaining leave days
                (
                    (TIMESTAMPDIFF(YEAR, emp.start_date, CURDATE()) + 1) * 18
                    -
                    (
                        SELECT 
                            COALESCE(SUM(DATEDIFF(approved_end_date, approved_start_date) + 1), 0)
                        FROM 
                            leave_requests lr2
                        WHERE 
                            lr2.employee = emp.employee_code
                            AND lr2.leaave_state = 1
                    )
                ) AS available_days

            FROM
                leave_requests lr
            LEFT JOIN
                employees emp ON lr.employee = emp.employee_code
            LEFT JOIN
                employees em ON lr.operator = em.employee_code
            LEFT JOIN
                leave_allocations la ON lr.leave_id = la.leave_id
            WHERE
                lr.leaave_state = 0
            ORDER BY 
                requested_start_date";
        $smtp = $this->link->prepare($sql);
        $smtp->execute();
        $data = $smtp->fetchAll(PDO::FETCH_ASSOC);
        return $data;
    }

    public function leaveDetails($leave_id) {
        $sql = "SELECT
                    leave_name,
                    leave_type,
                    leave_id
                FROM 
                    leave_allocations
                WHERE
                    leave_id = :leave_id
        ";

        try {
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':leave_id', $leave_id, PDO::PARAM_INT);
            $stmt->execute();
            
            // Fetch a single row as associative array
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            // Optional: Log the error
            // error_log('leaveDetails error: ' . $e->getMessage());
            return false;
        }
    }


    public function getLeaves(){
        $sql = "SELECT 
            leave_id,
            leave_name,
            leave_type
            FROM
                leave_allocations
            ORDER BY 
                leave_name ASC
            ";
        $smtp = $this->link->prepare($sql);
        $smtp->execute();
        $data = $smtp->fetchAll(PDO::FETCH_ASSOC);
        return $data;
    }

    public function disapproveLeaveRequest($requestId, $reason, $leave_state, $disapprovedBy, $disapprovedDate){
        $sql = "UPDATE leave_requests 
                SET 
                    leaave_state = :leave_state, 
                    reason = :reason,
                    disapproved_by = :disapprovedBy, 
                    disapproved_date = :disapprovedDate 
                WHERE 
                    id = :requestId";

        $stmt = $this->link->prepare($sql);
        return $stmt->execute([':leave_state' => $leave_state, ':reason' => $reason, ':disapprovedBy' => $disapprovedBy, ':disapprovedDate' => $disapprovedDate, ':requestId' => $requestId]);
    }


    public function getLeaveSlipData($leave_id) {
    // SQL query to get the leave slip details
    $sql = "SELECT 
                lr.id,
                er.fullname,
                lr.employee,
                lr.start_date,
                la.leave_type,
                lr.end_date,
                lr.requested_numdays,
                lr.approved_start_date,
                lr.approved_end_date,
                lr.approved_numdays,
                lr.leaave_state,
                e.fullname as operator,
                lr.disapproved_date,
                lr.datecaptured,
                lr.reason,
                COALESCE(emp.fullname,'Default Account') as approver,
                COALESCE(em.fullname,'Default Account') as disapprover
            FROM 
                leave_requests lr
            LEFT JOIN
                employees emp ON lr.approved_by = emp.employee_code
            LEFT JOIN
                employees em ON lr.disapproved_by = em.employee_code
            LEFT JOIN
                employees e ON lr.operator = e.employee_code       
            LEFT JOIN
                employees er ON lr.employee = er.employee_code            
            LEFT JOIN
                leave_allocations la ON lr.leave_id = la.leave_id
            WHERE 
                lr.id = :leave_id";  // Replace `leave_table` with the actual table name

    // Prepare the SQL statement
    $stmt = $this->link->prepare($sql);
    
    // Bind the leave_id parameter
    $stmt->bindParam(':leave_id', $leave_id, PDO::PARAM_INT);
    
    // Execute the query
    $stmt->execute();

    // Fetch the result as an associative array
    $result = $stmt->fetch(PDO::FETCH_ASSOC);

    // Check if the result exists
    if ($result) {
        return $result;
    } else {
        return null;  // Return null if no record found
    }
}


    public function getLeaveData() {
    $sql = "SELECT 
                e.employee_code,
                e.fullname,
                e.start_date,
                COALESCE(SUM(DATEDIFF(lr.approved_end_date, lr.approved_start_date) + 1), 0) AS total_taken,
                COUNT(lr.id) AS leave_requests_count
            FROM 
                employees e
            LEFT JOIN 
                leave_requests lr 
                ON lr.employee = e.employee_code
                AND lr.leave_type = 'Annual Leave'
                AND lr.leaave_state = 1
            WHERE
                e.employee_state = 1
            GROUP BY 
                e.employee_code, e.fullname, e.start_date
            ORDER BY 
                e.fullname ASC";

    $stmt = $this->link->prepare($sql);
    $stmt->execute();
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $result = [];

    foreach ($rows as $row) {
        $startDate = new DateTime($row['start_date']);
        $now = new DateTime();

        // Probation ends 3 months after start date
        $probationEnd = (clone $startDate)->modify('+3 months');

        // Leave accrual only starts after probation
        if ($probationEnd > $now) {
            $monthsWorked = 0;
        } else {
            $monthsWorked = ($now->diff($probationEnd)->y * 12) + $now->diff($probationEnd)->m;
        }

        $accruedDays = round($monthsWorked * 1.5, 1); // 1.5 days per full month
        $totalTaken = (float) $row['total_taken'];
        $totalLeft = max(0, $accruedDays - $totalTaken);

        $yearsWorked = $startDate->diff($now)->y + 1;

        $result[] = [
            'employee_code' => $row['employee_code'],
            'fullname' => $row['fullname'],
            'start_date' => $row['start_date'],
            'years_worked' => $yearsWorked,
            'total_days' => $accruedDays,
            'total_taken' => $totalTaken,
            'total_left' => $totalLeft,
            'leave_requests_count' => $row['leave_requests_count']
        ];
    }

    return $result;
}


    public function getLeaveDetails($employee_code) {
        $sql = "SELECT 
                    lr.id,
                    la.leave_name as leave_type,
                    lr.start_date,
                    lr.end_date,
                    lr.leaave_state,
                    lr.datecaptured,
                    COALESCE(emp.fullname,'Default Account') as operator,
                    CASE 
                        WHEN lr.leaave_state = 1 THEN 'Approved Leave'
                        WHEN lr.leaave_state = 0 THEN 'Requested Leave'
                        WHEN lr.leaave_state = 3 THEN 'Disapproved Leave'
                        ELSE 'Unknown Leave State'
                    END as leave_status,    
                    DATEDIFF(lr.end_date, lr.start_date) + 1 AS total_days
                FROM 
                    leave_requests lr
                LEFT JOIN
                    leave_allocations la ON lr.leave_id = la.leave_id
                LEFT JOIN
                    employees emp ON lr.operator = emp.employee_code
                WHERE
                    lr.employee = :employee_code
                    AND lr.leave_type = 'Annual Leave'
                ORDER BY
                    lr.start_date DESC";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':employee_code', $employee_code, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }


    //Return Unassigned Employees
    public function getbranchManagers() {
        $sql = "SELECT 
                emp.fullname,
                emp.phone_number,
                emp.current_location,
                emp.employee_code
            FROM 
                employees emp
            LEFT JOIN 
                posts pp ON emp.post = pp.id
            WHERE 
                emp.employee_state = 1
                AND emp.employee_code NOT IN (
                    SELECT shop_manager FROM branches WHERE shop_manager IS NOT NULL
                )";
    
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
    
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function checkPost($postname) {
        $sql = "SELECT COUNT(*) as count FROM posts WHERE postname = :postname";
    
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':postname', $postname, PDO::PARAM_STR);
        $stmt->execute();
    
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result['count']; // returns 0 if not found, >0 if exists
    }

    public function checkDepartment($departmentname) {
        $sql = "SELECT COUNT(*) as count FROM departments WHERE department_name = :departmentname";
    
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':departmentname', $departmentname, PDO::PARAM_STR);
        $stmt->execute();
    
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result['count']; // returns 0 if not found, >0 if exists
    }

    public function getDepartments() {
        $sql = "SELECT d.id, 
                    d.datecaptured, 
                    d.department_name,
                    COALESCE(emp.fullname,'Default System Administrator') as operator
                FROM 
                    departments d
                LEFT JOIN
                    employees emp ON d.operator = emp.employee_code ";
    
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
    
        return $stmt->fetchAll(PDO::FETCH_ASSOC); // Returns an array of departments
    }

    public function saveDepartment($departmentName, $datecaptured, $operator) {
        $sql = "INSERT INTO departments(department_name, datecaptured, operator) VALUES (:departmentName, :datecaptured, :operator)";
    
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':departmentName', $departmentName, PDO::PARAM_STR);
        $stmt->bindParam(':datecaptured', $datecaptured, PDO::PARAM_STR);
        $stmt->bindParam(':operator', $operator, PDO::PARAM_INT);
    
        return $stmt->execute();
    }

    public function savePost($postName, $departmentId, $gross_salary, $overtime_rate, $datecaptured, $operator) {
        $sql = "INSERT INTO posts (postname, department, gross_salary, overtime_rate, datecaptured, operator) 
                VALUES (:postname, :department, :gross_salary, :overtime_rate, :datecaptured, :operator)";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':postname', $postName);
        $stmt->bindParam(':department', $departmentId);
        $stmt->bindParam(':gross_salary', $gross_salary);
        $stmt->bindParam(':overtime_rate', $overtime_rate);
        $stmt->bindParam(':datecaptured', $datecaptured);
        $stmt->bindParam(':operator', $operator);
        return $stmt->execute();
    }

    public function getPosts(){
        $sql = "SELECT
                    p.id,
                    p.postname,
                    d.department_name,
                    p.gross_salary,
                    p.overtime_rate
                FROM 
                    posts p
                LEFT JOIN
                    departments d ON p.department = d.id
                WHERE
                    p.post_state = 1
                ORDER BY
                    p.postname ASC";  // Corrected the JOIN condition
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function checkEmailAddress($email){
        $sql = "SELECT COUNT(*) AS email_count FROM employees WHERE email_address = :email_address";
        $smtp = $this->link->prepare($sql);
        $smtp->bindParam('email_address', $email, PDO::PARAM_STR);
        $smtp->execute();

        // Fetch the count directly
        $result = $smtp->fetch(PDO::FETCH_ASSOC);
        return $result['email_count']; // Return the count of records
    }


    public function checkNationalID($national_id){
        $sql = "SELECT COUNT(*) AS national_id_count FROM employees WHERE national_id_number = :national_id";
        $smtp = $this->link->prepare($sql);
        $smtp->bindParam('national_id', $national_id, PDO::PARAM_STR);
        $smtp->execute();

        // Fetch the count directly
        $result = $smtp->fetch(PDO::FETCH_ASSOC);
        return $result['national_id_count']; // Return the count of records
    }


    public function saveEmployee($data) {
        $sql = "INSERT INTO employees 
            (employee_code, national_id_number, fullname, phone_number, email_address, gender, date_of_birth, current_location, home_village, home_district, nationality, post, start_date, contract_type, gross_salary, end_date, datecaptured, operator, employee_state)
            VALUES 
            (:employee_code, :national_id, :fullname, :phone_number, :email_address, :gender, :date_of_birth, :current_location, :home_village, :home_district, :nationality, :post, :start_date, :contract_type, :gross_salary, :end_date, :datecaptured, :operator, :employee_state)";
    
        try {
            $stmt = $this->link->prepare($sql);
            return $stmt->execute([
                ':employee_code'    => $data['employee_code'],
                'national_id'       => $data['national_id'],
                ':fullname'         => $data['fullname'],
                ':phone_number'     => $data['phone_number'],
                ':email_address'    => $data['email_address'],
                ':gender'           => $data['gender'],
                ':date_of_birth'    => $data['date_of_birth'],
                ':current_location' => $data['current_location'],
                ':home_village'     => $data['home_village'],
                ':home_district'    => $data['home_district'],
                ':nationality'      => $data['nationality'],
                ':post'             => $data['post'],
                ':start_date'       => $data['start_date'],
                ':contract_type'    => $data['contract_type'],
                ':gross_salary'     => $data['gross_salary'],
                ':end_date'         => $data['end_date'],
                ':datecaptured'     => $data['datecaptured'],
                ':operator'         => $data['operator'],
                ':employee_state'   => $data['employee_state']
            ]);
        } catch (PDOException $e) {
            // Log or handle error as needed
            return false;
        }
    }

    public function getNextEmployeeCode() {
        $sql = "SELECT MAX(employee_code) AS max_code FROM employees";
        try {
            $stmt = $this->link->query($sql);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
            if ($result && $result['max_code']) {
                return (int)$result['max_code'] + 1;
            } else {
                return 6000; // Starting code if none exists
            }
        } catch (PDOException $e) {
            // Log error if needed
            return 6000;
        }
    }

    public function updateEmployeeSalary($employee_code, $new_salary) {
        $sql = "UPDATE employees SET gross_salary = :new_salary WHERE employee_code = :employee_code AND employee_state = 1"; 

        try {
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':new_salary', $new_salary, PDO::PARAM_STR);
            $stmt->bindParam(':employee_code', $employee_code, PDO::PARAM_INT);

    
            return $stmt->execute();
        } catch (PDOException $e) {
            return false;
        }
    }

    public function emailExistsForOtherEmployee($email, $currentEmployeeCode) {
        $query = "SELECT COUNT(*) as total FROM employees WHERE email_address = :email AND employee_code != :code";
        $stmt = $this->link->prepare($query);
        $stmt->bindParam(':email', $email, PDO::PARAM_STR);
        $stmt->bindParam(':code', $currentEmployeeCode, PDO::PARAM_INT);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        return $result['total'] > 0;
    }

    public function nationalIDExistsForOtherEmployee($national_id, $currentEmployeeCode) {
        $query = "SELECT COUNT(*) as total FROM employees WHERE national_id_number = :national_id AND employee_code != :code";
        $stmt = $this->link->prepare($query);
        $stmt->bindParam(':national_id', $national_id, PDO::PARAM_STR);
        $stmt->bindParam(':code', $currentEmployeeCode, PDO::PARAM_INT);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        return $result['total'] > 0;
    }

    public function updateEmployeeDetails(array $data): bool {
        $sql = "UPDATE employees SET 
                    national_id_number = :national_id_number,
                    fullname = :fullname,
                    phone_number = :phone_number,
                    email_address = :email_address,
                    gender = :gender,
                    date_of_birth = :date_of_birth,
                    current_location = :current_location,
                    nationality = :nationality,
                    home_village = :home_village,
                    home_district = :home_district
                WHERE employee_code = :employee_code";

        $stmt = $this->link->prepare($sql);

        // Bind parameters
        $stmt->bindParam(':national_id_number', $data['national_id_number']);
        $stmt->bindParam(':fullname', $data['fullname']);
        $stmt->bindParam(':phone_number', $data['phone_number']);
        $stmt->bindParam(':email_address', $data['email_address']);
        $stmt->bindParam(':gender', $data['gender']);
        $stmt->bindParam(':date_of_birth', $data['date_of_birth']);
        $stmt->bindParam(':current_location', $data['current_location']);
        $stmt->bindParam(':nationality', $data['nationality']);
        $stmt->bindParam(':home_village', $data['home_village']);
        $stmt->bindParam(':home_district', $data['home_district']);
        $stmt->bindParam(':employee_code', $data['employee_code']);

        // Execute and return success/failure
        return $stmt->execute();
    }




    public function getEmployeeDetails($code) {
        $stmt = $this->link->prepare("SELECT 
                emp.national_id_number,
                emp.employee_code,
                emp.fullname,
                emp.phone_number,
                emp.email_address,
                emp.gender,
                emp.date_of_birth,
                emp.current_location,
                emp.home_village,
                emp.home_district,
                emp.nationality AS nationality_id,
                emp.post,
                emp.start_date,
                CASE
                    WHEN emp.employee_state = 1 THEN 'Active Employee'
                    ELSE 'Suspended'
                END as employee_status,
                emp.contract_type,
                emp.gross_salary,
                emp.end_date,
                emp.datecaptured,
                emp.operator AS operator_code,
                emp.picture,
                emp.employee_state,
                ll.location_name,
                pp.postname,
                dd.department_name,
                cc.nationality,
                d.district AS district_name,
                COALESCE(em.fullname, 'Default System Administrator') AS operator_name
    
            FROM 
                employees emp 
            LEFT JOIN 
                locations ll ON emp.current_location = ll.loc_id 
            LEFT JOIN 
                posts pp ON emp.post = pp.id
            LEFT JOIN 
                departments dd ON pp.department = dd.id
            LEFT JOIN 
                countries cc ON emp.nationality = cc.id
            LEFT JOIN 
                districts d ON ll.loc_dist = d.id
            LEFT JOIN 
                employees em ON emp.operator = em.employee_code 
            WHERE 
                emp.employee_code = ?");
        
        $stmt->execute([$code]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    
    
    
    

}