<?php

include_once 'config.php';

class users_model
{
    public $link;

    public function __construct()
    {
        $db_connection = new dbConnection();
        $this->link = $db_connection->connect();
    }

    public function updateLocation($loc_id, $value){
        $sql = "UPDATE locations SET location_state=:value WHERE id=:loc_id";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':value', $value, PDO::PARAM_INT);  
        $stmt->bindParam(':loc_id', $loc_id, PDO::PARAM_STR);
    
        
        try {
            $stmt->execute();
            return true;  
        } catch (PDOException $e) {
            return false;  // Return false if update fails
        }
    }

    public function updateUserState(string $username, int $newState): bool{
        $sql = "UPDATE users SET user_state = :newState WHERE username = :username";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':newState', $newState, PDO::PARAM_INT);
        $stmt->bindParam(':username', $username, PDO::PARAM_STR);

        if ($stmt->execute()) {
            return $stmt->rowCount() > 0;
        }

        return false;
    }

    public function isValidUser($employeeCode) {
        $sql = "SELECT COUNT(*) FROM users WHERE employee_code = ?";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([$employeeCode]);
        return $stmt->fetchColumn() > 0;
    }


    

    public function getuserDetails($username) {
        $query = "SELECT
                u.employee_code,
                emp.phone_number,
                emp.email_address,
                emp.current_location as loc,
                emp.home_district as hom_dis,
                la.location_name,
                ll.location_name as assigned_location,
                emp.home_village,
                emp.gender,
                CASE
                    WHEN u.role = 'SysAdmin' THEN 'System Administrator'
                    WHEN u.role = 'Administrator' THEN 'Chief Administrator'
                    ELSE u.role
                END as role,
                u.username,
                d.district as home_district,
                CASE 
                    WHEN u.employee_code IN(2000, 2001) THEN 'Super User'
                    WHEN emp.fullname IS NULL THEN 'UNKNOWN'
                    ELSE emp.fullname
                END AS fullname,
                CASE 
                    WHEN u.avatar IS NULL THEN 'default_avatar.png'
                    ELSE u.avatar
                END AS avatar,
                COALESCE(ll.login_count, 0) AS login_count
            FROM 
                users u
            LEFT JOIN
                employees emp ON u.employee_code = emp.employee_code
            LEFT JOIN
                locations la ON emp.current_location = la.loc_id
            LEFT JOIN
                locations ll ON u.location = ll.loc_id
            LEFT JOIN
                districts d ON emp.home_district = d.id
            LEFT JOIN
                (
                    SELECT username, COUNT(*) AS login_count
                    FROM logins
                    GROUP BY username
                ) ll ON u.username = ll.username
               WHERE
               		u.username = :username";
        
        $stmt = $this->link->prepare($query);
        $stmt->bindParam(':username', $username, PDO::PARAM_STR);
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function usernameExists($username){
        $sql = "SELECT id FROM users WHERE username = ? LIMIT 1";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([$username]);

        return $stmt->rowCount() > 0;
    }


    public function getAllSystemUsers() {
        $sql = "SELECT 
                ss.employee_code,
                ss.username,
                ss.role,
                ss.user_state,
                emp.fullname, 
                emp.email_address, 
                CASE
                    WHEN ss.role = 'Administrator' THEN 'Super User'
                    ELSE ss.role
                END as role,
                COALESCE(w.warehouse_name, b.branch_name, 'Powertex Ltd') as location_name,
                CASE
                    WHEN ss.user_state = 1 THEN 'Active'
                    WHEN ss.user_state = 0 THEN 'Inactive'
                END as user_status
            FROM 
                users ss
            LEFT JOIN
                employees emp ON ss.employee_code  = emp.employee_code
            LEFT JOIN
                warehouses w ON ss.location = w.warehouse_id
            LEFT JOIN
                branches b ON ss.location = b.branch_id
            WHERE
                ss.employee_code NOT IN(2000)
            ORDER BY
                fullname ASC";
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getaccDetails($username) {
        $query = "SELECT
                emp.phone,
                emp.email,
                emp.emplocation as loc,
                emp.home_district as hom_dis,
                la.location_name,
                emp.home_village,
                emp.gender,
                u.password as current_password,
                d.district as home_district,
                CASE 
                    WHEN u.employee_code = 2001 THEN 'Super User'
                    WHEN emp.fullname IS NULL THEN 'UNKNOWN'
                    ELSE emp.fullname
                END AS fullname,
                CASE 
                    WHEN emp.avatar IS NULL THEN 'default_avatar.png'
                    ELSE emp.avatar
                END AS avatar,
                COALESCE(ll.login_count, 0) AS login_count
            FROM 
                users u
            LEFT JOIN
                employees emp ON u.employee_code = emp.employee_id
            LEFT JOIN
                locations la ON emp.emplocation = la.loc_id
            LEFT JOIN
                districts d ON emp.home_district = d.id
            LEFT JOIN
                (
                    SELECT username, COUNT(*) AS login_count
                    FROM logins
                    GROUP BY username
                ) ll ON u.username = ll.username
               WHERE
               		u.username = :username";
        
        $stmt = $this->link->prepare($query);
        $stmt->bindParam(':username', $username, PDO::PARAM_STR);
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function checkemployee($employee_code) {
        try {
            
            if ($employee_code === 2000) {
                return true;
            }
    
            
            $stmt = $this->link->prepare('SELECT employee_code FROM employees WHERE employee_code = :emp_code');
            $stmt->execute(array(':emp_code' => $employee_code));
            
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($result !== false && isset($result['employee_code'])) {
                return true;
            } else {
                return false;
            }
        } catch (PDOException $e) {
            echo "Database error: " . $e->getMessage();
            return false;
        }
    }

    public function isBranchLocation($location) {
        $query = "SELECT COUNT(*) as count FROM branches WHERE branch_id = :location";
        $stmt = $this->link->prepare($query);
        $stmt->bindParam(':location', $location);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result['count'] > 0;
    }

    public function isWarehouseLocation($location) {
        $query = "SELECT COUNT(*) as count FROM warehouses WHERE warehouse_id = :location";
        $stmt = $this->link->prepare($query);
        $stmt->bindParam(':location', $location);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result['count'] > 0;
    }



    public function employeeRoleExists($employee_code, $system_role, $location) {
        try {
            $stmt = $this->link->prepare(
                'SELECT 
                    COUNT(*) 
                FROM 
                    users 
                WHERE 
                    employee_code = :employee_code 
                    AND role = :system_role
                    AND location = :location'
            );
            $stmt->execute([
                ':employee_code' => $employee_code,
                ':system_role' => $system_role, ':location' =>$location
            ]);

            return $stmt->fetchColumn() > 0;
        } catch (PDOException $e) {
            // Log or handle error as needed
            return false;
        }
    }

    public function validateEmployeeLocation($employee_code, $system_role, $location) {
        try {
            // Now validate based on role
            if ($system_role === 'Supervisor') {
                $stmt = $this->link->prepare('SELECT COUNT(*) 
                    FROM 
                        warehouses 
                    WHERE 
                        supervisor = :employee_code 
                        AND location = :location
                        AND warehouse_state = 1'
                );
            } elseif ($system_role === 'Cashier') {
                $stmt = $this->link->prepare(
                    'SELECT COUNT(*) 
                    FROM 
                        branches 
                    WHERE 
                        shop_manager = :employee_code 
                        AND location = :location
                        AND branch_state = 1'
                );
            } else {
                // Other roles don’t require this check
                return true;
            }

            $stmt->execute([
                ':employee_code' => $employee_code,
                ':location' => $location
            ]);

            return $stmt->fetchColumn() > 0;

        } catch (PDOException $e) {
            // Log or handle the error
            return false;
        }
    }


    

    public function changeAvatar($avatar, $employee_code) {
        try {
            $stmt = $this->link->prepare('UPDATE users SET avatar = :avatar WHERE employee_code =:employee_code');
            $stmt->bindParam(':avatar', $avatar, PDO::PARAM_STR);
            $stmt->bindParam(':employee_code', $employee_code);
            return $stmt->execute();
        } catch (PDOException $e) {
            echo "Database error: " . $e->getMessage();
            return false;
        }
    }
    
    public function superUserAvatar($avatar, $username) {
        try {
            $stmt = $this->link->prepare('UPDATE users SET avatar = :avatar WHERE username = :username');
            $stmt->bindParam(':avatar', $avatar);
            $stmt->bindParam(':username', $username);
            return $stmt->execute();
        } catch (PDOException $e) {
            echo "Database error: " . $e->getMessage();
            return false;
        }
    }

    public function getAdminAvatar($username) {
        try {
            // Prepare the SQL query to select the avatar for the given username
            $stmt = $this->link->prepare('SELECT avatar FROM users WHERE username = :username');
            $stmt->bindParam(':username', $username, PDO::PARAM_STR);
            $stmt->execute();
            
            // Fetch the result
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Check if the avatar is found and return it, otherwise return a default avatar
            if ($result !== false && isset($result['avatar'])) {
                return $result['avatar'];
            } else {
                // Return a default avatar if none is found
                return 'default_avatar.png';
            }
        } catch (PDOException $e) {
            // Handle database errors
            echo "Database error: " . $e->getMessage();
            return 'default_avatar.png';
        }
    }

    public function updateUserDetails($employee_code, $fullname, $phone, $email, $location_from, $home_village, $home_district, $gender) {
        $sql = "UPDATE
                    employees
                SET 
                    fullname = :fullname,
                    phone = :phone,
                    email = :email,
                    current_location = :location_from,
                    home_village = :home_village,
                    home_district = :home_district,
                    gender = :gender
                WHERE
                    employee_code = :employee_code";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':fullname', $fullname);
        $stmt->bindParam(':phone', $phone);
        $stmt->bindParam(':email', $email);
        $stmt->bindParam(':location_from', $location_from);
        $stmt->bindParam(':home_village', $home_village);
        $stmt->bindParam(':home_district', $home_district);
        $stmt->bindParam(':gender', $gender);
        $stmt->bindParam(':employee_code', $employee_code);

        return $stmt->execute();
    }

    public function updatePass($username, $password) {
        $sql = "UPDATE
                    users
                SET 
                    password = :password
                WHERE
                    username = :username";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':username', $username);
        $stmt->bindParam(':password', $password);
        return $stmt->execute();
    }
    public function getAllHashedPins() {
        $query = "SELECT user_pin FROM users";
        $stmt = $this->link->prepare($query);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_COLUMN);
    }

    public function getAllUsersAndHashedPins() {
        // SQL to retrieve user details along with hashed PINs
        $query = "SELECT 
                    u.id, 
                    u.employee_code, 
                    u.username, 
                    u.role, 
                    u.user_state,  
                    u.password, 
                    u.user_pin,
                    COALESCE(emp.fullname, 'Super User') as employee_name
                FROM 
                    users u
                LEFT JOIN
                    employees emp ON u.employee_code = emp.employee_id ";
        $stmt = $this->link->prepare($query);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC); // Return user data with hashed PINs
    }

    public function getPasswordByUsername($username) {
        $sql = "SELECT password FROM users WHERE username = :username";
        $stmt = $this->link->prepare($sql);

        if ($stmt === false) {
            // Handle error in preparing the statement
            error_log("SQL Error: " . $this->link->errorInfo());
            return null;
        }
        
        // Bind the username parameter to the prepared statement
        $stmt->bindParam(':username', $username, PDO::PARAM_STR); 
        $stmt->execute();
        
        // Fetch the result
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result) {
            return $result['password'];
        } else {
            return null; // No user found with that username
        }
    }

    public function getUsers(){
        $sql = "SELECT 
                us.employee_code as code,
                emp.fullname as employee,
                us.username,
                us.role,
                us.user_state,
                us.id as usertbid,
                us.employee_code,
                CASE
                    WHEN us.user_state = 1 THEN 'Active'
                    WHEN us.user_state = 0 THEN 'Deactivated'
                END as status 
            FROM
                users us
            LEFT JOIN
                employees emp ON us.employee_code = emp.employee_id
            WHERE
                us.employee_code NOT IN(2000, 2001, 2002)
            ORDER BY 
                emp.fullname ASC
           ";
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
        // Check if the result is not empty
        if (!empty($result)) {
            return $result;
        } else {
            return []; // Return an empty array if no districts are found
        }

    }
    

    public function checkSession($session_id) {
        try {
             $sql = "SELECT COUNT(*) AS session_count FROM logins WHERE session_id = :session_id";
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':session_id', $session_id, PDO::PARAM_STR);
            $stmt->execute();
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            return $result['session_count'];
        } catch (PDOException $e) {
            error_log("Error checking session: " . $e->getMessage());
            return 0; // Return 0 in case of error
        }
    }
    
    public function checkuser($username) {
        try {
             $sql = "SELECT COUNT(*) AS num_users FROM users WHERE username = :username";
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':username', $username, PDO::PARAM_STR);
            $stmt->execute();
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            return $result['num_users'];
        } catch (PDOException $e) {
            error_log("Error checking session: " . $e->getMessage());
            return 0; // Return 0 in case of error
        }
    }
    
    public function authenticateUser($username) {
        try {
            $sql = 'SELECT
                u.employee_code,
                COALESCE(emp.fullname, "Super User") as employee_name,
                u.role, 
                u.username,
                u.user_state,
                u.password
            FROM 
                users u
            LEFT JOIN
                employees emp ON u.employee_code = emp.employee_code
            WHERE 
                username = :username';
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':username', $username, PDO::PARAM_STR);
            //$stmt->bindParam(':password', $password, PDO::PARAM_STR);
            $stmt->execute();
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            return $user ? $user : false;
        } catch (PDOException $e) {
            error_log("Error authenticating user: " . $e->getMessage());
            return false;
        }
    }

    public function getLoginData() {
        $sql = "SELECT *, 
                CONCAT((SELECT fullname FROM employees AS d WHERE r.emp_id = d.employee_id), account) AS emp_id,
                IF(
                    (SELECT COUNT(username) FROM logins AS f WHERE attempt = 1 AND r.emp_id = f.emp_id AND r.account = f.account) > 0,
                    (SELECT COUNT(username) FROM logins AS f WHERE attempt = 1 AND r.emp_id = f.emp_id AND r.account = f.account),
                    '0'
                ) AS success,
                IF(
                    (SELECT COUNT(username) FROM logins AS f WHERE attempt = 0 AND r.emp_id = f.emp_id AND r.account = f.account) > 0,
                    (SELECT COUNT(username) FROM logins AS f WHERE attempt = 0 AND r.emp_id = f.emp_id AND r.account = f.account),
                    '0'
                ) AS failed,
                COUNT(username) AS total
            FROM 
                logins AS r
            WHERE 
                emp_id NOT IN(2000, 2001, 2002, 2003, 2004, 2005, 2006)
            GROUP BY 
                account, emp_id
            ORDER BY date DESC
            LIMIT 15";
        
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC); // Return the result as an associative array
    }

    public function countLocationsByName($loc_name) {
        $sql = "SELECT * FROM locations WHERE location_name = :loc_name";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':loc_name', $loc_name);
        $stmt->execute();
        
        // Return the number of rows that match the query
        return $stmt->rowCount();
    }

    public function addlocation($code, $location_name, $dist_id ){
        $sql = "INSERT INTO locations (loc_id, location_name, loc_dist, location_state) 
        VALUES (:loc_id, :location_name, :loc_dist, :location_state)";
        $location_state = 1; 
        $stmt = $this->link->prepare($sql);

        $stmt->bindParam(':loc_id', $code);
        $stmt->bindParam(':location_name', $location_name);
        $stmt->bindParam(':loc_dist', $dist_id);
        $stmt->bindParam(':location_state', $location_state);
        $stmt->execute();

    }

    public function getMaxLocationId() {
        $sql = "SELECT MAX(id) AS current FROM locations";
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return $result ? $result['current'] : 0;  // Return the max ID or 0 if no result
    }

    public function logLogin($session_id, $emp_id, $username, $date, $account, $attempt, $sess_state) {
        $sql = "INSERT INTO logins(session_id, emp_id, username, date, account, attempt, active_session)
                VALUES (:session_id, :emp_id, :username, :date, :account, :attempt, :sess_state )";

        $stmt = $this->link->prepare($sql);

        // Binding parameters to avoid SQL injection
        $stmt->bindParam(':session_id', $session_id);
        $stmt->bindParam(':emp_id', $emp_id);
        $stmt->bindParam(':username', $username);
        $stmt->bindParam(':date', $date);
        $stmt->bindParam(':account', $account);
        $stmt->bindParam(':attempt', $attempt);
        $stmt->bindParam(':sess_state', $sess_state);

        // Execute the query
        return $stmt->execute();
    }

    public function updateRole($username, $new_role) {
        try {
            $sql = "UPDATE users SET role = :userrole WHERE username = :username";
    
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':userrole', $new_role, PDO::PARAM_STR);
            $stmt->bindParam(':username', $username, PDO::PARAM_STR);
    
            if ($stmt->execute()) {
                return true; // Success
            } else {
                return false; // Failure
            }
        } catch (PDOException $e) {
            return 0; // Return failure on exception
        }
    }
    public function checkUsernameExists($user, $table_id) {
        $sql = "SELECT * FROM users WHERE username = :username AND id != :table_id";
        
        try {
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':username', $user, PDO::PARAM_STR);
            $stmt->bindParam(':table_id', $table_id, PDO::PARAM_INT);
            $stmt->execute();

            // Fetch the number of rows
            return $stmt->rowCount();
        } catch (PDOException $e) {
            // Handle any PDO errors
            throw new Exception("Database query error: " . $e->getMessage());
        }
    }

    public function getRoles() {
        $sql = "SELECT 
                r.*, 
                (SELECT fullname FROM employees AS f WHERE r.employee_code = f.employee_id) AS employee_name
            FROM
                users AS r 
            WHERE
                user_state = 1 AND employee_code NOT IN (2000,2001, 2002, 2003, 2004, 2005, 2006)
            ORDER BY
                CASE 
                    WHEN r.role = 'Finance' THEN 0
                    WHEN r.role = 'Stockman' THEN 1
                    WHEN r.role = 'Cashier' THEN 2
                    WHEN r.role = 'Supervisor' THEN 3
                    WHEN r.role = 'Waiter' THEN 4
                    ELSE 5
                END";

        $stmt = $this->link->prepare($sql);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC); // Return the result as an associative array
    }

    public function getEmployees() {
        $sql = "SELECT 
                employee_id,
                fullname
            FROM
                employees 
            WHERE
                employee_state = 1
            ORDER BY 
                fullname ASC";

        $stmt = $this->link->prepare($sql);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC); // Return the result as an associative array
    }

    public function updateEmployee($fullname, $position, $department, $email, $phone, $location, $home_village, $district, $gender, $employee_code){
        $sql = "UPDATE employees SET fullname =:fullname, position =:post, department =:department, email =:email, phone =:phone, emplocation =:location, home_village =:home_village, home_district =:district, gender =:gender WHERE employee_id =:employee_code";
        try {
            $smtp = $this->link->prepare($sql);
            $smtp->bindParam(':fullname',$fullname, PDO::PARAM_STR);
            $smtp->bindParam(':post',$position, PDO::PARAM_STR);
            $smtp->bindParam(':department',$department, PDO::PARAM_STR);
            $smtp->bindParam(':email',$email, PDO::PARAM_STR);
            $smtp->bindParam(':phone',$phone, PDO::PARAM_STR);
            $smtp->bindParam(':location',$location, PDO::PARAM_STR);
            $smtp->bindParam(':home_village',$home_village, PDO::PARAM_STR);
            $smtp->bindParam(':district',$district, PDO::PARAM_INT);
            $smtp->bindParam(':gender',$gender, PDO::PARAM_STR);
            $smtp->bindParam(':employee_code',$employee_code, PDO::PARAM_INT);

            $smtp->execute();

            return 1;
        } catch (PDOException $e) {
            // Handle any PDO errors
            throw new Exception("Database query error: " . $e->getMessage());
        }
    }
    
    public function employeeState($state,  $employee_code){
        $sql = "UPDATE employees SET employee_state =:state WHERE employee_id =:employee_code";
        try {
            $smtp = $this->link->prepare($sql);
            
            $smtp->bindParam(':state', $state, PDO::PARAM_INT);
            $smtp->bindParam(':employee_code', $employee_code, PDO::PARAM_INT);

            $smtp->execute();

            return 1;
        } catch (PDOException $e) {
            // Handle any PDO errors
            throw new Exception("Database query error: " . $e->getMessage());
        }
    }


    
    public function getEmployeeData($employee_id) {
        $sql = "SELECT 
                emp.employee_id,
                emp.fullname,
                emp.phone,
                emp.email,
                emp.emplocation,
                ll.location_name,
                emp.department,
                emp.position,
                emp.home_district,
                emp.home_village,
                emp.gender,
                dd.district as home_district_name,
                emp.employee_state,
                CASE 
                    WHEN emp.employee_state = 1 THEN 'ACTIVE'
                    WHEN emp.employee_state = 0 THEN 'SUSPENDED'
                    ELSE 'UNKNOWN STATE'
                END as state
            FROM
                employees  emp
            LEFT JOIN
                locations ll ON emp.emplocation = ll.loc_id
            LEFT JOIN
                districts dd ON emp.home_district = dd.id
            WHERE
                emp.employee_id = :employee_id";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':employee_id', $employee_id, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetch(PDO::FETCH_ASSOC); // Return the result as an associative array
    }

    public function createUser($data) {
        $sql = "INSERT INTO users (employee_code, username, password, role, user_state, `2fa`, phone_number, location
                ) VALUES (
                    :employee_code, :username, :password, :role, :user_state, :fa2, :phone_number, :location
                )";

        $stmt = $this->link->prepare($sql);

        $stmt->bindParam(':employee_code', $data['employee_code'], PDO::PARAM_INT);
        $stmt->bindParam(':username', $data['username'], PDO::PARAM_STR);
        $stmt->bindParam(':password', $data['password'], PDO::PARAM_STR);
        $stmt->bindParam(':role', $data['system_role'], PDO::PARAM_STR);
        $stmt->bindValue(':user_state', 1, PDO::PARAM_INT); // Hardcoded active state
        $stmt->bindValue(':fa2', 1, PDO::PARAM_INT); // Default 2FA off
        $stmt->bindParam(':phone_number', $data['phone_number'], PDO::PARAM_STR);
        $stmt->bindParam(':location', $data['location'], PDO::PARAM_STR);

        return $stmt->execute();
    }


    public function updateroles($username, $columnName, $state) {
        $allowedColumns = ['reports', 'user_state', 'cards', 'cash_trans', 'membership', 'users', 'customers', 'employees', 'stock'];

        if (!in_array($columnName, $allowedColumns)) {
            return "Error: Invalid column name.";
        }

        $sql = "UPDATE users SET $columnName = :state WHERE username = :username";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':state', $state, PDO::PARAM_INT);
        $stmt->bindParam(':username', $username, PDO::PARAM_STR);

        if ($stmt->execute()) {
            return true;
        } else {
            return false;
        }
    }

    public function cancelActive($username){
        $sql = "UPDATE logins SET active_session = 0 WHERE username = :username";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':username', $username, PDO::PARAM_STR);
        
        if($stmt->execute()){
            return true;
        }else{
            return false;
        }
    }

    public function logoutUpdates($logout_time, $session_id){
        $sql = "UPDATE logins SET logout_time = :logout_time WHERE session_id = :session_id";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':logout_time', $logout_time, PDO::PARAM_STR);
        $stmt->bindParam(':session_id', $session_id, PDO::PARAM_STR);
        
        if($stmt->execute()){
            return true;
        }else{
            return false;
        }
    }

    public function updateUserPassword($username, $hashed_password) {
        $stmt = $this->link->prepare("UPDATE users SET password = ? WHERE username = ?");
        return $stmt->execute([$hashed_password, $username]);
    }

    public function changePassword($username, $password){
        $sql = "UPDATE users SET password = :password WHERE username = :username";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':username', $username, PDO::PARAM_STR);
        $stmt->bindParam(':password', $password, PDO::PARAM_STR);

        $stmt->execute();
    }

    public function getPosts(){
        $sql = "SELECT postname, department FROM posts";
        $smtp = $this->link->prepare($sql);
        $smtp->execute();

        $posts = $smtp->fetchAll(PDO::FETCH_ASSOC);
        return $posts;
    }
    
    
    public function addPost($postdata) {
        try {
            // Prepare the SQL statement
            $sql = "INSERT INTO posts (postname, department)
                    VALUES (:postname, :department)";

            $stmt = $this->link->prepare($sql);

            // Bind the parameters
            $stmt->bindParam(':postname', $postdata['postname']);
            $stmt->bindParam(':department', $postdata['department']);

            // Execute the statement
            $stmt->execute();

            return true;
        } catch (PDOException $e) {
            echo "Error: " . $e->getMessage();
            return false;
        }
    }

    public function isWarehouseSupervisor($employee_code) {
        $stmt = $this->link->prepare("SELECT COUNT(*) FROM warehouses WHERE supervisor = ?");
        $stmt->execute([$employee_code]);
        return $stmt->fetchColumn() > 0;
    }

    public function isBranchManager($employee_code) {
        $stmt = $this->link->prepare("SELECT COUNT(*) FROM branches WHERE shop_manager = ?");
        $stmt->execute([$employee_code]);
        return $stmt->fetchColumn() > 0;
    }

    public function locationHasWarehouse($location_id) {
        $stmt = $this->link->prepare("SELECT COUNT(*) FROM warehouses WHERE location = ?");
        $stmt->execute([$location_id]);
        return $stmt->fetchColumn() > 0;
    }

    public function locationHasBranch($location_id) {
        $stmt = $this->link->prepare("SELECT COUNT(*) FROM branches WHERE location = ?");
        $stmt->execute([$location_id]);
        return $stmt->fetchColumn() > 0;
    }



}